package romanconvertorlogic;

/**
 * <p>Deze klasse bevat statische methoden om de omzetting van en naar het Romeinse cijferstelsel te berekenen</p>
 * <ul>
 *   <li>een ingegeven Arabisch nummer (van 1 tot 4999) vertalen naar zijn Romeinse voorstelling.</li>
 *   <li>een ingegeven Romeins nummer (van I tot MMMMCMXCIX) vertalen naar zijn Arabische voorstelling.</li>
 * </ul>
 *
 * @author Kristien Van Assche
 * @version 5 januari 2009
 */
public class RomanConvertorLogic {
	/** geldige romeinse cijfers */
	private static final String ROMEINSE_LETTERS = "MDCLXVI. ";

	/** symbolische constante voor het Arabisch cijferstelsel */
	private static final int ARABISCH = 0;

	/** symbolische constante voor het Romeins cijferstelsel */
	private static final int ROMEINS = 1;

	/** symbolische constante voor een ongeldig cijferstelsel */
	private static final int FOUT = 2;

	/** geldige romeinse cijfers, gebruik makend van een tweedimensionale rij */
	private static final String[][] ROMEINSE_GETALLEN = { 	{"M", "MM",	"MMM",	"MMMM",	"V.M ",	"VI.M ","VII.M ", "VIII.M ","IX.M "	},
							{"C", "CC",	"CCC",	"CD",	"D",	"DC",	"DCC"	, "DCCC",	"CM" 	},
							{"X", "XX",	"XXX",	"XL",	"L",	"LX",	"LXX"	, "LXXX",	"XC"	},
							{"I", "II",	"III",	"IV",	"V",	"VI",	"VII"	, "VIII",	"IX"	} };

	/**
	 * Van hieruit wordt de gepaste conversie methode opgeroepen.
	 *
	 * @param ingave de tekst die moet vertaald worden naar het andere cijferstelsel
	 * @return de waarde volgens het andere cijferstelsel
	 *
	 * @see #bepaalCijferStelsel(String)
	 * @see Omzettingen#vertaalNaarRomeinsStelsel(int)
	 * @see Omzettingen#vertaalNaarArabischStelsel(String)
	 */
	public static String translate(String input) {
			int stelsel = bepaalCijferStelsel(input);

			switch (stelsel) {
				case ARABISCH :
					return vertaalNaarRomeinsStelsel(Integer.parseInt(input));
				case ROMEINS :
					int waarde = vertaalNaarArabischStelsel(input);
					return "" + (waarde > 0 ? waarde : "Ongeldige ingave");
				default:
					return "Ongeldige ingave";
			}
	}

	/**
	 * Bepaalt of het om een getal in het Arabisch dan wel het Romeins stelsel gaat.
	 *
	 * @param ingave de tekst waarvan het cijferstelsel
	 * @return int geeft aan of het cijferstelsel Arabisch(0), Romeins(1) of ongeldig(2) is
	 */
	private static int bepaalCijferStelsel(String ingave) {
		boolean arabisch = true;
		boolean romeins = true;

		// doorloop ingave op zoek naar ongeldige karakters
		for (int i = 0; i < ingave.length(); i++) {
			char c = ingave.charAt(i);

			if (c < '0' || c > '9' && c != '+' && c != ' ') {
				arabisch = false;
			}
			if (ROMEINSE_LETTERS.indexOf(c) == -1 && c != '+' && c != ' ') {
				romeins = false;
			}
		}

		return arabisch ? ARABISCH : romeins ? ROMEINS : FOUT;
	}

	/**
	 * Methode die een Arabisch getal naar het Romeinse stelsel omzet.
	 *
	 * @param i de Arabische waarde
	 * @return String de Romeinse equivalent-waarde
	 */
	private static String vertaalNaarRomeinsStelsel(int i) {
		String romeinseWaarde = "";

		int duizendtal = i / 1000;
		if (duizendtal != 0) {
			romeinseWaarde += ROMEINSE_GETALLEN[0][duizendtal - 1];
		}

		int honderdtal = (i % 1000 - i % 100) / 100;
		if (honderdtal != 0) {
			romeinseWaarde += ROMEINSE_GETALLEN[1][honderdtal - 1];
		}

		int tiental = (i % 100 - i % 10) / 10;
		if (tiental != 0) {
			romeinseWaarde += ROMEINSE_GETALLEN[2][tiental - 1];
		}

		int eenheid = i % 10;
		if (eenheid != 0) {
			romeinseWaarde += ROMEINSE_GETALLEN[3][eenheid - 1];
		}

		return romeinseWaarde;
	}

	/**
	 * Methode die een Romeins getal naar het Arabisch stelsel omzet.
	 *
	 * @param s de Romeinse waarde
	 * @return int de Arabische equivalent-waarde
	 */
	private static int vertaalNaarArabischStelsel(String s) {
		int arabischeWaarde = 0;

                for (int i = 0; i < ROMEINSE_GETALLEN.length; i++) {
			for (int j = ROMEINSE_GETALLEN[0].length - 1; j >= 0; j--) {
                                s = s.trim();
                                
				if (s.startsWith(ROMEINSE_GETALLEN[i][j].trim())) {
					arabischeWaarde += (j+1) * Math.pow(10, 3-i);
					s = s.substring(ROMEINSE_GETALLEN[i][j].trim().length());
					break;
				}
			}
		}

		if (s.length() > 0) {
			return -1;
		}
		else {
			return arabischeWaarde;
		}
	}
}

